 /*
  * vdev_root_sys_block.c
  * Copyright (C) 2025  Aitor C.Z. <aitor_czr@gnuinos.org>
  * 
  * This program is free software: you can redistribute it and/or modify it
  * under the terms of the GNU General Public License as published by the
  * Free Software Foundation, either version 3 of the License, or
  * (at your option) any later version.
  * 
  * This program is distributed in the hope that it will be useful, but
  * WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  * See the GNU General Public License for more details.
  * 
  * You should have received a copy of the GNU General Public License along
  * with this program.  If not, see <http://www.gnu.org/licenses/>.
  * 
  * See the COPYING file.
  */


#include "libvdev/sglib.h"
#include "libvdev/misc.h"
#include "libvdev/sbuf.h"

#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <dirent.h>
#include <mntent.h>


int main(int argc, char *argv[])
{
    FILE *fstab = NULL;
    struct mntent *e;
    char *buf = NULL;
    DIR *dirp;
    struct dirent *E;
    sbuf_t sys;
    sbuf_t mountpoint;
    sbuf_t rootdev;
    const char *str;
    const char *dir = "/sys/block";
    const char *mtab = "/etc/mtab";
        
    fstab = setmntent(mtab, "r");
    if (!fstab)
    {
        fprintf(stderr, "setmntent(): error trying to open /etc/mtab: '%s'\n", strerror(errno));
        return -errno;
    }
    
    sbuf_init(&sys);        
    sbuf_init(&mountpoint);        
    sbuf_init(&rootdev);
     
    while ((e = getmntent(fstab))) {
        if (sys.len > 0 && mountpoint.len > 0 && rootdev.len > 0)
            break;
        else if (sys.len == 0 && !strcmp(e->mnt_type, "sysfs"))
            sbuf_addstr(&sys, e->mnt_dir);
        else if (mountpoint.len == 0 && !strcmp(e->mnt_fsname, "udev") && !strcmp(e->mnt_type, "devtmpfs"))
            sbuf_addstr(&mountpoint, e->mnt_dir);
        else if (!strcmp(e->mnt_dir, "/"))
            sbuf_addstr(&rootdev, e->mnt_fsname);
    }
    
    str = strstr(rootdev.buf, mountpoint.buf) + strlen(mountpoint.buf) + 1;
    if (!strncmp(str, "mapper/", strlen("mapper/"))) {
        sbuf_free(&sys);
        sbuf_free(&mountpoint);
        sbuf_free(&rootdev);
        return 1;
    }
            
    dirp = opendir(dir);
    if (!dirp) {
        fprintf(stderr, "error opening current directory: %s\n", strerror(errno));
        sbuf_free(&sys);
        sbuf_free(&mountpoint);
        sbuf_free(&rootdev);
        return -errno;
    };

    while ((errno=0, E=readdir(dirp))) {
        if (strstr(E->d_name, ".") || strstr(E->d_name, ".."))
            continue; 
        if (!strncmp(str, E->d_name, strlen(E->d_name))) {
            sbuf_t path;
            sbuf_init(&path);
            sbuf_concat(&path, 6, dir, "/", E->d_name, "/", str, "/partition");
                    
            // Does '/sys/block/{E->d_name}/{e->mnt_fsname}/partition' exist?
            if(access(path.buf, F_OK) == 0) {
                printf("%s\n", E->d_name);
                sbuf_free(&path);
                break;
            }
            sbuf_free(&path);
        }
    }
    
    closedir(dirp);
    sbuf_free(&sys);
    sbuf_free(&mountpoint);
    sbuf_free(&rootdev);
    
    return 0;
}
