 /*
  * vdev_reset_usb_devices.c
  * Copyright (C) 2025  Aitor C.Z. <aitor_czr@gnuinos.org>
  * 
  * This program is free software: you can redistribute it and/or modify it
  * under the terms of the GNU General Public License as published by the
  * Free Software Foundation, either version 3 of the License, or
  * (at your option) any later version.
  * 
  * This program is distributed in the hope that it will be useful, but
  * WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  * See the GNU General Public License for more details.
  * 
  * You should have received a copy of the GNU General Public License along
  * with this program.  If not, see <http://www.gnu.org/licenses/>.
  * 
  * See the COPYING file.
  */


#include "libvdev/sglib.h"
#include "libvdev/util.h"
#include "libvdev/misc.h"
#include "libvdev/sbuf.h"

#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <mntent.h>

const char *progname = "vdev_reset_usb_devices";
const char *mtab = "/etc/mtab";

typedef char *cstr;

// prototypes
SGLIB_DEFINE_VECTOR_PROTOTYPES (cstr);
SGLIB_DEFINE_VECTOR_FUNCTIONS (cstr);

// free a list of cstr vectors
// always succeeds
static int vdev_cstr_vector_free_all(struct sglib_cstr_vector *vec)
{
    // free all strings
    for (int i = 0; i < sglib_cstr_vector_size(vec); i++) {
        if (sglib_cstr_vector_at(vec, i) != NULL) {
            free(sglib_cstr_vector_at(vec, i));
            sglib_cstr_vector_set(vec, NULL, i);
        }
    }
    return 0;
}

static int get_mounted_devices(struct sglib_cstr_vector *mounted_devices)
{
    int rc;
    DIR *dirp;
    struct dirent *E;
    struct mntent *e;
    FILE *fstab = NULL;   
    struct sglib_cstr_vector devices;
    const char *dir = "/sys/block";
    
    dirp = opendir(dir);
    if (!dirp) {
        vdev_error ("%s error opening current directory: %s\n", progname, strerror(errno));
        return -errno;
    };
    
    sglib_cstr_vector_init (&devices);

    while ((errno=0, E=readdir(dirp))) { 
        if (!strstr(E->d_name, ".") && !strstr(E->d_name, "..")) {
            char *copy = vdev_strdup_or_null(E->d_name);
            if (!copy)
                return -ENOMEM;
            rc = sglib_cstr_vector_push_back(&devices, copy);
            if (rc != 0) {
                free(copy);
                vdev_cstr_vector_free_all(&devices);       
                sglib_cstr_vector_free(&devices);
                return rc;
            }    
        }
    }
    if (errno) {
        vdev_error ("%s error reading current directory: %s\n", progname, strerror(errno));
        closedir(dirp);
        vdev_cstr_vector_free_all(&devices);       
        sglib_cstr_vector_free(&devices);
        return -errno;
    }
    closedir(dirp);
  
    fstab = setmntent(mtab, "r");
    if (!fstab) {
        vdev_error("%s: setmntent(): error trying to open /etc/mtab: '%s'\n", progname, strerror (errno));
        vdev_cstr_vector_free_all(&devices);       
        sglib_cstr_vector_free(&devices);
        return -errno;
    }
     
    while ((e = getmntent(fstab))) {
        if (!strncmp(e->mnt_fsname, "/dev/", 5)) {
            const char *str = strstr(e->mnt_fsname, "/dev/") + 5;
            for (int i = 0; i < sglib_cstr_vector_size(&devices); i++) {
                if (!strncmp(str, sglib_cstr_vector_at(&devices, i), strlen(sglib_cstr_vector_at(&devices, i)))) {
                    bool already_exists = false;
                    for (int j = 0; j < sglib_cstr_vector_size(mounted_devices); j++) {
                        if (!strcmp(sglib_cstr_vector_at(&devices, i), sglib_cstr_vector_at(mounted_devices, j))) {
                            already_exists = true;
                            break;
                        }
                    }
                    if (!already_exists) {
                        char *aux = vdev_strdup_or_null(sglib_cstr_vector_at(&devices, i));
                        if (!aux) {
                            endmntent(fstab);
                            vdev_cstr_vector_free_all(&devices);       
                            sglib_cstr_vector_free(&devices);
                            return -ENOMEM;
                        }
                        int rc = sglib_cstr_vector_push_back(mounted_devices, aux);
                        if (rc < 0) {
                            endmntent(fstab);
                            free(aux);
                            vdev_cstr_vector_free_all(&devices);       
                            sglib_cstr_vector_free(&devices);
                            return rc;
                        }
                    }
                }
            }
        }
    }
    endmntent(fstab);
    vdev_cstr_vector_free_all(&devices);       
    sglib_cstr_vector_free(&devices);
    return rc;
}

int main(int argc, char *argv[])
{
    int rc = 0;
    DIR *dirp;
    struct dirent *E;   
    struct sglib_cstr_vector block_devices;
    struct sglib_cstr_vector usb_devices;
    struct sglib_cstr_vector mounted_devices;
    const char *blockdir = "/sys/dev/block";
    const char *usbdir = "/sys/bus/usb/drivers/usb";
    sbuf_t pid1uid;
    bool user_session_started = false;
    
    send_to_background();        
    
    sbuf_init(&pid1uid);
    while (!user_session_started) {
        dirp = opendir("/proc");
        if (!dirp) {
            vdev_error("%s: opendir(): %s\n", progname, strerror(errno));
            sbuf_free(&pid1uid);
            exit(EXIT_FAILURE);
        }
	
        while ((E = readdir(dirp))) {
            sbuf_t s;
            FILE *fp = NULL;
            char line[1024];
            if (!isdigit(E->d_name[0]))
                continue;
            sbuf_init(&s);
            sbuf_concat(&s, 3, "/proc/", E->d_name, "/loginuid");
            fp = fopen(s.buf, "r");
            if (!fp) {
                sbuf_free(&s);
                continue;
            }
            sbuf_free(&s);
            if (fgets(line, 1024, fp)) {
                line[strcspn(line, "\n")] = '\0';
                if (pid1uid.len == 0)
                    sbuf_addstr(&pid1uid, line);
                else if (strcmp(line, pid1uid.buf)) {
                    user_session_started = true;
                    fclose(fp);
                    break;
                }
            }
            fclose(fp);
        }	
        closedir(dirp);
	}
    sbuf_free(&pid1uid);
   
    if (access(mtab, F_OK) != 0)
        exit(EXIT_FAILURE);
    
    sglib_cstr_vector_init (&mounted_devices);
    rc = get_mounted_devices(&mounted_devices);
    if (rc != 0) {
        vdev_cstr_vector_free_all(&mounted_devices);       
        sglib_cstr_vector_free(&mounted_devices);
        exit(EXIT_FAILURE);
    } 
    
    dirp = opendir(blockdir);
    if (!dirp) {
        vdev_error ("%s error opening current directory: %s\n", progname, strerror(errno));
        vdev_cstr_vector_free_all(&mounted_devices);       
        sglib_cstr_vector_free(&mounted_devices);
        exit(EXIT_FAILURE);
    };

    sglib_cstr_vector_init (&block_devices);
    while ((errno=0, E=readdir(dirp))) {
        if (!strstr(E->d_name, ".") && !strstr(E->d_name, "..")) {
            char *copy = vdev_strdup_or_null(E->d_name);
            if (!copy) {
                rc = -ENOMEM;
                goto Free_block_devices;
            }
            rc = sglib_cstr_vector_push_back(&block_devices, copy);
            if (rc < 0) {
                free(copy);
                goto Free_block_devices;
            }
        }
    }
    if (errno) {
        vdev_error ("%s error reading current directory: %s\n", progname, strerror(errno));
        closedir(dirp);
        rc = -errno;
        goto Free_block_devices;
    }
    closedir(dirp);

    sglib_cstr_vector_init (&usb_devices);

    for (int i = 0; i < sglib_cstr_vector_size(&block_devices); i++) {
        sbuf_t sl;
        char buffer[PATH_MAX+1]={0};          
        char *token = NULL;
        char *tmp = NULL;
        bool is_mounted = false;
        
        sbuf_init(&sl);
        sbuf_concat(&sl, 3, blockdir, "/", sglib_cstr_vector_at(&block_devices, i));
        
        rc = readlink(sl.buf, buffer, sizeof(buffer));
        if (rc <0) { 
            vdev_error("readlink(%s) error", sl.buf);
            sbuf_free(&sl);
            goto Free_usb_devices;
        }
        
        tmp = rindex(buffer, '/')+1;
        if (!strncmp(tmp, "loop", 4) && isdigit(tmp[4])) {
            sbuf_free(&sl);
            continue;
        }
        
        for (int j = 0; j < sglib_cstr_vector_size(&mounted_devices); j++) {
            const char *str = sglib_cstr_vector_at(&mounted_devices, j);
            if (!strncmp(tmp, str, strlen(str))) {
                 is_mounted = true;
                 break; 
            }
        }
        if (is_mounted) {
            sbuf_free(&sl);
            continue;
        }
        token = strtok(strstr(buffer, "/devices/"), "/");
        while (token != NULL) {
            bool already_exists = false;
            if (isdigit(token[0])) {
                for (int j = 0; j < sglib_cstr_vector_size(&usb_devices); j++) {
                    if (!strcmp(token, sglib_cstr_vector_at(&usb_devices, j))) {
                        already_exists = true;
                        break;
                    }
                }
                if (!already_exists) {
                    char *copy = vdev_strdup_or_null(token);
                    if (!copy) {
                        sbuf_free(&sl);
                        rc = -ENOMEM;
                        goto Free_usb_devices;
                    }
                    rc = sglib_cstr_vector_push_back(&usb_devices, copy);
                    if (rc < 0) {
                        free(copy);
                        sbuf_free(&sl);
                        goto Free_usb_devices;
                    }
                }
            }
            token = strtok(NULL, "/");
        }
        sbuf_free(&sl);
    }
    
    dirp = opendir(usbdir);
    if (!dirp) {
        vdev_error ("%s error opening current directory: %s\n", progname, strerror(errno));
        rc = -errno;
        goto Free_usb_devices;
    };
 
    while ((errno=0, E=readdir(dirp))) {
        if (isdigit(E->d_name[0]) && !strstr(E->d_name, ".") && !strstr(E->d_name, "..")) {
            for (int i = 0; i < sglib_cstr_vector_size(&usb_devices); i++) {
                if (!strcmp(E->d_name, sglib_cstr_vector_at(&usb_devices, i))) {
                    FILE *pfin = NULL;
                    pid_t pid;
                    int wstatus;
                    sbuf_t cmd;
                    sbuf_init(&cmd);
                    sbuf_concat(&cmd, 3, "/bin/echo ", E->d_name, " |sudo tee /sys/bus/usb/drivers/usb/unbind >/dev/null");
                    pfin = epopen(cmd.buf, &pid);
                    if (pfin) { 
                        fclose(pfin);
                        waitpid(pid, &wstatus, 0);
                        sbuf_init(&cmd);
                        sbuf_concat(&cmd, 3, "/bin/echo ", E->d_name, " |sudo tee /sys/bus/usb/drivers/usb/bind >/dev/null");
                        pfin = epopen(cmd.buf, &pid);
                        if (pfin) { 
                            fclose(pfin);
                            waitpid(pid, &wstatus, 0);
                        }
                    }
                    sbuf_free(&cmd);
                }
            }
        }
    }
    
    goto Free_block_devices;
    
Free_usb_devices:    
    vdev_cstr_vector_free_all(&usb_devices);       
    sglib_cstr_vector_free(&usb_devices);
    
Free_block_devices:    
    vdev_cstr_vector_free_all(&block_devices);       
    sglib_cstr_vector_free(&block_devices);
    
    vdev_cstr_vector_free_all(&mounted_devices);       
    sglib_cstr_vector_free(&mounted_devices);
 
    return rc;
}
