/*
   nl_monitor.c
   Copyright (C) 2025  Aitor Cuadrado Zubizarreta <aitor_czr@gnuinos.org>

   This program is dual-licensed: you can redistribute it and/or modify
   it under the terms of the GNU General Public License version 3 or later as 
   published by the Free Software Foundation. For the terms of this 
   license, see LICENSE.GPLv3+ or <http://www.gnu.org/licenses/>.

   You are free to use this program under the terms of the GNU General
   Public License, but WITHOUT ANY WARRANTY; without even the implied 
   warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
   See the GNU General Public License for more details.

   Alternatively, you are free to use this program under the terms of the 
   Internet Software Consortium License, but WITHOUT ANY WARRANTY; without
   even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
   For the terms of this license, see LICENSE.ISC or 
   <http://www.isc.org/downloads/software-support-policy/isc-license/>.
*/

#ifndef _GNU_SOURCE
#define _GNU_SOURCE 1
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <fcntl.h>

#include "libudev.h"
#include "libudev-private.h"

// read, but mask EINTR
// return number of bytes read on success 
// return -errno on I/O error
ssize_t vdev_read_uninterrupted( int fd, char* buf, size_t len ) {
   
   ssize_t num_read = 0;
   
   if( buf == NULL ) {
      return -EINVAL;
   }
   
   while( (unsigned)num_read < len ) {
      ssize_t nr = read( fd, buf + num_read, len - num_read );
      if( nr < 0 ) {
         
         int errsv = -errno;
         if( errsv == -EINTR ) {
            continue;
         }
         
         return errsv;
      }
      if( nr == 0 ) {
         break;
      }
      
      num_read += nr;
   }
   
   return num_read;
}

// entry point
int main( int argc, char** argv )
{
   int rc = 0;
   int c = 0, i = 0;
   char event_buf[8192];
   struct udev* udev_client = NULL;
   struct udev_monitor* monitor = NULL;
   struct udev_device* dev = NULL;
   ssize_t buflen, bufpos;
   ssize_t nr = 0;
   
   char const* required_fields[] = {
      "\nSUBSYSTEM=",
      "\nDEVPATH=",
      NULL
   };
 
   // get the event 
   memset( event_buf, 0, 8192 );
   nr = vdev_read_uninterrupted( STDIN_FILENO, event_buf, 8192 );
   if( nr <= 0 ) {
      
      rc = -errno;
      fprintf( stderr, "[ERROR] %s: Failed to read event from stdin: %s\n", argv[0], strerror( -rc ) );
      exit(1);
   }
   
   // simple sanity check for requirements 
   for( int i = 0; required_fields[i] != NULL; i++ ) {
      
      if( strstr( event_buf, required_fields[i] ) == NULL ) {
         
         // head of line? with no leading '\n'?
         if( strncmp( event_buf, required_fields[i] + 1, strlen(required_fields[i]) - 1 ) != 0 ) {
            
            fprintf(stderr, "[ERROR] %s: Missing required field '%s'\n", argv[0], required_fields[i] + 1 );
            fprintf(stderr, "[ERROR] %s: Pass -h for a list of required fields\n", argv[0] );
            exit(1);
         }
      }
   }
   
   udev_client = udev_new();
   if( udev_client == NULL ) {
      
      // OOM
      exit(2);
   }
   
   monitor = udev_monitor_new_from_netlink( udev_client, NULL);
   if( monitor == NULL ) {
      
      // OOM or error
      udev_unref( udev_client );
      exit(2);
   }
        
   while( event_buf[i] != '\0' ) {
			
	  if( event_buf[i] == '\n' ) {
         event_buf[i] = '\0';
      }
      i++;
   }

   buflen = i+1;
   bufpos=strlen(event_buf) + 1;      
        
   if( strstr( event_buf, "@/" ) == NULL ) {
      
      // invalid header 
      printf("%s", "invalid message header: missing '@' directive" );
      udev_unref( udev_client );
      exit(3);
   } 
 
   dev = udev_device_new_from_nulstr( udev_monitor_get_udev(monitor), event_buf + bufpos, buflen - bufpos);
   if( dev == NULL ) {
      
      rc = -errno;
      printf("udev_device_new_from_buffer() rc = %d", rc);
      udev_unref( udev_client );    
      exit(4);
   }
   
   //udev_device_set_is_initialized(udev_device);

   /* send processed event back to libudev listeners */
   udev_monitor_send_device(monitor, NULL, dev);
                        
   udev_monitor_unref(monitor);
   monitor = NULL;
        
   udev_device_unref(dev);
   dev = NULL;
   udev_unref( udev_client );
   
   return 0;
}
