 /*
  * vdev_modprobe.c
  * Copyright (C) 2025  Aitor C.Z. <aitor_czr@gnuinos.org>
  * 
  * This program is free software: you can redistribute it and/or modify it
  * under the terms of the GNU General Public License as published by the
  * Free Software Foundation, either version 3 of the License, or
  * (at your option) any later version.
  * 
  * This program is distributed in the hope that it will be useful, but
  * WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  * See the GNU General Public License for more details.
  * 
  * You should have received a copy of the GNU General Public License along
  * with this program.  If not, see <http://www.gnu.org/licenses/>.
  * 
  * See the COPYING file.
  */


#include "libvdev/sglib.h"
#include "libvdev/config.h"
#include "libvdev/util.h"
#include "libvdev/param.h"
#include "libvdev/daemonlet.h"
#include "libvdev/misc.h"
#include "libvdev/sbuf.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdbool.h>
#include <mntent.h>
#include <sys/stat.h>
#include <sys/utsname.h>

#include <poll.h>
#include <linux/netlink.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <sys/types.h>
#include <signal.h>
#include <sys/signalfd.h>
#include <sys/fanotify.h>

const char *progname = "vdev_modprobe";
const char *pidfile ="/run/vdev_modprobe.pid";

static volatile sig_atomic_t linux_signal = -1;

static int fd = -1;
static struct flock fl;

const char *mtab = "/etc/mtab";
const char *mountpoint = "/dev";
static char sysfs_mountpoint[PATH_MAX + 1];

#define VDEV_LINUX_NETLINK_RECV_BUF_MAX 4097

/* Size of buffer to use when reading fanotify events */
#define FANOTIFY_BUFFER_SIZE 8192

typedef char *cstr;
SGLIB_DEFINE_VECTOR_PROTOTYPES(cstr);
SGLIB_DEFINE_VECTOR_FUNCTIONS(cstr);

struct sysfs_scan_context {
    char *uevent_path;
    struct sglib_cstr_vector *frontier;
};

/* Enumerated list of FDs to poll */
enum {
    FD_POLL_SIGNAL = 0,
    FD_POLL_FANOTIFY,
    FD_POLL_NETLINK,
    FD_POLL_MAX
};

// connection to the linux kernel for hotplug
struct vdev_linux_context {
    // netlink address
    struct sockaddr_nl nl_addr;
   
    // poll on multiple file descriptors
    struct pollfd pfd[FD_POLL_MAX];
};

// sigterm handler
static void sig_handler(int signum, siginfo_t *info, void *extra)
{
    linux_signal = signum;
    //pid_t sender_pid = info->si_pid;

    switch(signum) {
    case SIGALRM:
    case SIGHUP:
    case SIGTERM:
    case SIGINT:
    case SIGQUIT:
        break;
    default:
        break;
    }
}

/* Setup fanotify notifications (FAN) mask. All these defined in fanotify.h. */
static uint64_t event_mask =
    (FAN_OPEN |       /* File was opened */
    FAN_EVENT_ON_CHILD); /* We want to be reported of events in files of the directory */

static int initialize_fanotify()
{
    int fanotify_fd;

    /* Create new fanotify device */
    if ((fanotify_fd = fanotify_init(FAN_CLASS_NOTIF, O_RDONLY)) < 0) {
        vdev_error("Couldn't setup new fanotify device: %s\n", strerror(errno));
        return -errno;
    }

    /* Add new fanotify mark on the filesystem object */
    if (fanotify_mark(fanotify_fd, FAN_MARK_ADD | FAN_MARK_MOUNT, event_mask, AT_FDCWD, "/") < 0) {
        vdev_error("Couldn't add the mark mask on the filesystem object: '%s'\n", strerror(errno));
        close(fanotify_fd);
        return -errno;
    }

    return fanotify_fd;
}

static void shutdown_fanotify(int fanotify_fd)
{
    /* Remove the mark, using same event mask as when creating it */
    fanotify_mark(fanotify_fd, FAN_MARK_REMOVE, event_mask, AT_FDCWD, "/");
}

static int initialize_signals(void)
{
    int signal_fd = -1;
    struct sigaction sa;
   
    memset (&sa, 0, sizeof(sa));
          
    // initialize signal handling
    sa.sa_sigaction = &sig_handler;
    sa.sa_flags = SA_RESTART | SA_NODEFER | SA_SIGINFO; //  SA_RESETHAND | 
    sigemptyset(&sa.sa_mask);
    sigaction(SIGTERM, &sa, 0);
    sigaction(SIGINT, &sa, 0);
    sigaction(SIGQUIT, &sa, 0);
    sigaction(SIGALRM, &sa, 0);
    sigaction(SIGHUP, &sa, 0);

    /* We want to handle some signals in the signal_fd */
    sigaddset(&sa.sa_mask, SIGTERM);
    sigaddset(&sa.sa_mask, SIGALRM);
    sigaddset(&sa.sa_mask, SIGHUP);

    // Get new FD to read signals from it
    if ((signal_fd = signalfd(-1, &sa.sa_mask, 0)) < 0) {  // What happens in case of SFD_NONBLOCK??
        vdev_error("Couldn't setup signal FD: '%s'\n", strerror (errno));
        return -1;
    }

    return signal_fd;
}

// free a list of cstr vectors
// always succeeds
static int vdev_cstr_vector_free_all(struct sglib_cstr_vector *vec)
{
    // free all strings
    for (int i = 0; i < sglib_cstr_vector_size(vec); i++) {
        if (sglib_cstr_vector_at(vec, i) != NULL) {
            free(sglib_cstr_vector_at(vec, i));
            sglib_cstr_vector_set(vec, NULL, i);
        }
    }
    return 0;
}

// stop listening 
static void vdev_linux_context_free(struct vdev_linux_context *ctx)
{
    shutdown_fanotify(ctx->pfd[FD_POLL_FANOTIFY].fd);
    
    // shut down 
    if (ctx) {
        for (int i = 0; i < FD_POLL_MAX - 1; i++) {
            if (ctx->pfd[i].fd >= 0) {
                close(ctx->pfd[i].fd);
                ctx->pfd[i].fd = -1;
            }
        }
    }
}

// 'buf' must be freed after usage
static void get_sysfs_path(char **buf)
{
    struct mntent *e;
    FILE *fstab = NULL;
   
    fstab = setmntent(mtab, "r");
    if (!fstab) {
        vdev_error("%s: setmntent(): error trying to open /etc/mtab: '%s'\n", progname, strerror (errno));
        exit(EXIT_FAILURE);
    }
    
    *buf = (char*)malloc(sizeof(char) * 32);
    if (!*buf) {
        vdev_error ("%s: Memory allocation failure: '%s'\n", progname, strerror (errno));
        exit(EXIT_FAILURE);
    }
   
    *buf[0] = '\0';
     
    while ((e = getmntent(fstab))) {
        if (!strcmp(e->mnt_type, "sysfs")) {
            sprintf(*buf, "%s", e->mnt_dir);
            break;
        }
    }
   
    endmntent(fstab);
}

// make the full sysfs path from the dev path, plus an additional path 
// return NULL on OOM
static char *vdev_linux_sysfs_fullpath(char const *sysfs_mountpoint, char const *devpath, char const *attr_path)
{
    char *tmp = NULL;
    char *ret = NULL;
   
    tmp = vdev_fullpath(sysfs_mountpoint, devpath, NULL);
    if (!tmp)
        return NULL;
   
    ret = vdev_fullpath(tmp, attr_path, NULL);
    free(tmp);
   
    return ret;
}

// read the kernel-given device subsystem from sysfs 
// return 0 on success, and set *subsystem
// return -ENOMEM on OOM 
// return negative on readlink failure
static int vdev_linux_sysfs_read_subsystem(const char *mountpoint, char const *devpath, char **subsystem)
{
    int rc = 0;
    char linkpath[PATH_MAX+1];
    size_t linkpath_len = PATH_MAX;
    char *subsystem_path = NULL;
   
    memset(linkpath, 0, PATH_MAX+1);
   
    subsystem_path = vdev_linux_sysfs_fullpath(mountpoint, devpath, "subsystem");
    if (subsystem_path == NULL)
        return -ENOMEM;
    
    if (access(subsystem_path, F_OK) != 0) {
        /* directory doesn't exist */
        free(subsystem_path);
        return 0;
    }
    
   
    rc = readlink(subsystem_path, linkpath, linkpath_len);
    if (rc < 0) {
        vdev_error("%s: readlink('%s') %s\n", progname, subsystem_path, strerror(errno));
        free(subsystem_path);
        return -errno;
    }
   
    free(subsystem_path);
   
    *subsystem = vdev_basename(linkpath, NULL);
    if (*subsystem == NULL)
        return -ENOMEM;
   
    return 0;
}

// scan a directory in /sys/devices directory, to find its child directories, that are pushed back to scan_ctx->frontier
// return 0 on success
// return -ENOMEM on OOM
// return -errno on failure to stat
static int scan_device_directory(char const *fp, void *cls)
{
    struct sysfs_scan_context *scan_ctx = (struct sysfs_scan_context*)cls;
   
    struct sglib_cstr_vector *frontier = scan_ctx->frontier;

    int rc = 0;  
    struct stat sb;
    char *fp_base = NULL;
    char *fp_dup = NULL;
   
    fp_base = rindex (fp, '/') + 1;
   
    if (fp_base == NULL)
        return 0;
   
    // skip . and .. 
    if (strcmp(fp_base, ".") == 0 || strcmp(fp_base, "..") == 0)
        return 0;
  
    // add directories
    rc = lstat (fp, &sb);
    if (rc != 0) {
        vdev_error ("%s: lstat('%s'): '%s'\n", progname, fp, strerror(errno));
        return -errno;
    }
   
    if (!S_ISDIR(sb.st_mode) && strcmp(fp_base, "uevent") != 0)
        return 0;
   
    fp_dup = vdev_strdup_or_null(fp);
    if (fp_dup == NULL)
        return -ENOMEM;
   
    if (S_ISDIR(sb.st_mode)) {
        rc = sglib_cstr_vector_push_back(frontier, fp_dup);
        if (rc != 0) {
            vdev_error ("%s: sglib_cstr_vector_push_back('%s'): '%s'\n", progname, fp_dup, strerror(errno));
            free (fp_dup);
            return rc;
        }
    /* this is a uevent; this directory is a device */      
    } else {
        scan_ctx->uevent_path = fp_dup;
    }
   
    return 0;
}

/**
 * \brief Variadic function
 */
static int find_devices_at_frontier(char *sysfs_mountpoint, const char *device_frontier, struct sglib_cstr_vector *uevent_paths)
{
    int rc = 0;
    struct sglib_cstr_vector frontier;
    struct sysfs_scan_context scan_ctx;

    sglib_cstr_vector_init (&frontier);
   
    memset(&scan_ctx, 0, sizeof(struct sysfs_scan_context));
   
    scan_ctx.frontier = &frontier;

    rc = vdev_load_all(device_frontier, scan_device_directory, &scan_ctx);
    if (rc != 0) {
        vdev_error ("%s: vdev_load_all('%s'): '%s'\n", progname, device_frontier, strerror(errno));
        vdev_cstr_vector_free_all(&frontier);
        sglib_cstr_vector_free(&frontier);      
        return rc;
    }
   
    while (1) {
        int len = sglib_cstr_vector_size (&frontier);
        if (len == 0)
            break;
   
        char *dir = sglib_cstr_vector_at (&frontier, len - 1);
        sglib_cstr_vector_set(&frontier, NULL, len - 1);
      
        sglib_cstr_vector_pop_back(&frontier);
      
        // scan for more devices 
        rc = vdev_load_all(dir, scan_device_directory, &scan_ctx);
        if (rc != 0) {
            vdev_error("%s: vdev_load_all('%s'): '%s'\n", progname, dir, strerror(errno));
            free (dir);
            break;
        }
      
        // is one of them a uevent?
        if (scan_ctx.uevent_path) {
            const char *str;
            bool is_ok = false;
     
            char *uevent_path = vdev_strdup_or_null(scan_ctx.uevent_path + strlen(sysfs_mountpoint));
            if (!uevent_path) {
                free (dir);
                free (scan_ctx.uevent_path);
                scan_ctx.uevent_path = NULL;
                rc = -ENOMEM;
                break;
            }

            rc = sglib_cstr_vector_push_back(uevent_paths, uevent_path);
            if (rc < 0) {
                vdev_error ("%s: sglib_cstr_vector_push_back('%s'): '%s'\n", progname, uevent_path, strerror(errno));
                free(uevent_path);
                free(dir);
                free(scan_ctx.uevent_path);
                scan_ctx.uevent_path = NULL;
                break;
            }
     
            free (scan_ctx.uevent_path);
            scan_ctx.uevent_path = NULL;
        }
      
        free (dir);
    }

    vdev_cstr_vector_free_all(&frontier);
    sglib_cstr_vector_free(&frontier);

    return rc;
}

static int find_devices(char *sysfs_mountpoint, struct sglib_cstr_vector *uevent_paths)
{
    int rc = 0;
    struct sglib_cstr_vector frontier;
    struct sysfs_scan_context scan_ctx;
   
    memset(&scan_ctx, 0, sizeof(struct sysfs_scan_context));
      
    char *devroot = NULL;

    sglib_cstr_vector_init (&frontier);
   
    scan_ctx.frontier = &frontier;
   
    devroot = vdev_fullpath(sysfs_mountpoint, "/devices", NULL);     // devroot <= /sys/devices
    if (devroot == NULL)      
        return -ENOMEM;
    
    rc = vdev_load_all(devroot, scan_device_directory, &scan_ctx);
    if (rc != 0) {
        vdev_error ("%s: vdev_load_all('%s'): '%s'\n", progname, devroot, strerror(errno)); 
        free (devroot);
        vdev_cstr_vector_free_all(&frontier);   
        sglib_cstr_vector_free(&frontier);      
        return rc;
    }
   
    free(scan_ctx.uevent_path);
    scan_ctx.uevent_path = NULL;
 
    for (int i = 0; i < sglib_cstr_vector_size(&frontier); i++)
        find_devices_at_frontier(sysfs_mountpoint, sglib_cstr_vector_at(&frontier, i), uevent_paths);
    
    vdev_cstr_vector_free_all(&frontier);
    sglib_cstr_vector_free(&frontier);
    
    if (devroot != NULL) {
        free(devroot);
        devroot = NULL;
    }
   
    return rc;
}

// get a uevent from a uevent file 
// replace newlines with '\0', making the uevent look like it came from the netlink socket
// (i.e. so it can be parsed by vdev_linux_parse_request)
// return 0 on success
// return -ENOMEM on OOM
// return -errno on failure to stat or read
static int vdev_read_uevent(char const *fp_uevent, char **ret_uevent_buf, size_t *ret_uevent_len)
{
    int rc = 0;
    struct stat sb;
    char *uevent_buf = NULL;
    size_t uevent_buf_len = 0;
    size_t uevent_len = 0;
   
    // get uevent size  
    rc = stat(fp_uevent, &sb);
    if (rc != 0) {
        vdev_error ("%s: stat('%s'): '%s'\n", progname, fp_uevent, strerror (errno));
        return -errno;
    }
    else
        uevent_buf_len = sb.st_size;
   
    // read the uevent as long as it's a regular file
    if (fp_uevent != NULL && (sb.st_mode & S_IFMT) == S_IFREG) {
        uevent_buf = VDEV_CALLOC(char, uevent_buf_len);
        if (uevent_buf == NULL)
            return -ENOMEM;
      
        rc = vdev_read_file(fp_uevent, uevent_buf, uevent_buf_len);
        if (rc != 0) {
            // failed in this 
            vdev_error ("%s: read_file('%s'): '%s'\n", progname, fp_uevent, strerror (errno));
            free(uevent_buf);
        }
        else {
            for (unsigned int i = 0; i < uevent_buf_len; i++) {
                if (uevent_buf[i] == '\n')           
                    uevent_buf[i] = '\0';
            }
     
            // NOTE: the stat size is an upper-bound.  Find the exact number of bytes.
            for (uevent_len = 0; uevent_len < uevent_buf_len;) {
                if (*(uevent_buf + uevent_len) == '\0')
                    break;
                uevent_len += strlen(uevent_buf + uevent_len) + 1;
            }
      
            *ret_uevent_buf = uevent_buf;
            *ret_uevent_len = uevent_len;
        }
    }
   
    return rc;
}

// carry out the modprobe command by sending it to the running daemonlet.
// restart the daemonlet if we need to (e.g. if it hasn't been started, or it died since the last device request).
// return the daemonlet's exit status on success (will be positive if the daemonlet failed in error).
// return -ENOMEM on OOM
// return -EPERM if the daemonlet could not be started, or was not responding and we could not restart it.  A subsequent call probably won't succeed.
// return -EAGAIN if the daemonlet was not responding, but we were able to stop it.  A subsequent call might succeed in starting it up and feeding it the request.
static int feed_modprobe_daemonlet(char const *fp_uevent,
                   struct vdev_config *config,
                   struct vdev_daemonlet *daemonlet,
                   const char *name,
                   const char *command,
                   int *error_fd)
{
    int rc = 0;
    struct stat sb;
    char *uevent_buf = NULL;
    size_t uevent_buf_len = 0;
    char *full_devpath = NULL;
    sbuf_t s;     
    char **env = NULL;
    size_t num_env = 2;
    int64_t daemonlet_rc = 0;         
   
    // get uevent    
    sbuf_init(&s);
    sbuf_concat(&s, 2, sysfs_mountpoint, fp_uevent);
    rc = vdev_read_uevent(s.buf, &uevent_buf, &uevent_buf_len);
    if (rc != 0) {
        vdev_error ("%s: read_uevent('%s'): '%s'\n", progname, fp_uevent, strerror(errno));
        return rc;
    }
    sbuf_free(&s);
   
    if (uevent_buf_len == 0) {
        free(uevent_buf);
        return 0;
    }             
     
    env = VDEV_CALLOC(char*, num_env + 1);    
    rc = vdev_make_env_str("VDEV_DAEMONLET", "1", &env[0]);
    if (rc != 0) {
        VDEV_FREE_LIST(env);
        return rc;
    } 
  
    for (unsigned int i = 0; i < uevent_buf_len;) {
        char *tmp = rindex(uevent_buf + i, '=') + 1;
        if (!strncmp(uevent_buf+i, "MODALIAS=", strlen("MODALIAS=")) && strlen(tmp) > 0) {
            char *tmp = rindex(uevent_buf + i, '=') + 1;
            if (strlen(tmp) > 2) {
                int num_attempts = 0;
                rc = vdev_make_env_str("VDEV_OS_MODALIAS", tmp, &env[1]);
                if (rc != 0) {
                    VDEV_FREE_LIST(env);
                    return rc;
                }
                /* try twice, in case we need to stop and start it */
                while (num_attempts < 2) {
                    rc = vdev_daemonlet_send_command(env, 
                                     num_env,
                                     name,
                                     false,
                                     &daemonlet_rc,
                                     daemonlet);
                    if (rc != 0) {
                        vdev_error("vdev_action_daemonlet_send_command('%s') rc = %d\n", name, rc);
                        if (rc == -EAGAIN) {
                            rc = vdev_daemonlet_stop(daemonlet, name);
                            if (rc < 0) {
                                vdev_error("vdev_action_daemonlet_stop('%s', PID=%d) rc = %d\n", 
                                       name, daemonlet->pid, rc);
                                rc = -EPERM;
                                break;
                            } else {
                                rc = vdev_daemonlet_start(config,
                                              daemonlet,
                                              command,
                                              name,
                                              false,
                                              error_fd);
                                if (rc < 0) {
                                    vdev_error("vdev_action_daemonlet_start('%s') rc = %d\n",
                                           name, rc);
                                    rc = -EPERM;
                                    break;
                                } else {  
                                    num_attempts++;
                                    continue;
                                }
                            }
                        } else {
                            rc = -EPERM;
                            break;
                        }
                    } else {
                        break;
                    }
                }
            }
        }
        i += strlen(uevent_buf + i) + 1;
    }
    
    VDEV_FREE_LIST(env);

    if (rc == 0) {
        vdev_debug("daemonlet '%s' returned %d\n", name, (int)daemonlet_rc);
        return (int)daemonlet_rc;
    } else {
        return rc;
    }
}

static bool dmi_decode_type_is_tower_or_desktop()
{    
    char line[1024];
    const char *cmd = "/usr/sbin/dmidecode -t chassis | /bin/grep Type | /usr/bin/cut -d' ' -f2";
    FILE *fp = NULL;
    bool res = false;
    
    fp = popen(cmd, "r");
    if (fp == NULL) /* ENOSYS */
        return false;

    if (fgets(line, 1024, fp)) {
        line[strcspn(line, "\n")] = '\0';
        if (!strcmp(line, "Tower") || !strcmp(line, "Desktop"))
            res = true;
    }
    pclose(fp);
    return res;
}

int main(int argc, char **argv)
{
    int rc;
    sbuf_t s;
    char *buf = NULL;
    struct vdev_config *config;
    struct sglib_cstr_vector uevents;
    int error_fd = 0;
    bool async = false;
    struct vdev_daemonlet *daemonlet;
    const char *command = "/lib/vdev/modprobe.sh";
    const char *name = "/etc/vdev/actions/002-modalias.act";
    const char *config_file = "/etc/vdev/vdevd.conf";
    bool background = false;
    ssize_t nr = 0;
    struct vdev_linux_context ctx;
    
    FILE *pfin = NULL;
    pid_t pid;
    int wstatus;
    char *cmd = NULL;
   
    /* netlink */
    size_t slen =  VDEV_LINUX_NETLINK_RECV_BUF_MAX;
    int so_passcred_enable = 1;
    
    if (argc > 1 && (!strcmp(argv[1], "-b") || !strcmp(argv[1], "--background")))
        background = true;           
   
    if (access(mtab, F_OK) != 0) {
        snprintf (sysfs_mountpoint, PATH_MAX, "/sys"); 
    } else {
        get_sysfs_path (&buf);
        if (buf && buf[0] != '\0') {
            snprintf (sysfs_mountpoint, PATH_MAX, buf);       
            free (buf);   
        } else {
            vdev_error ("%s: Cannot get sysfs path\n", progname);
            exit(1);
        }
    }

    rc = vdev_pidfile_write(pidfile);
    if (rc != 0) {
        vdev_error("vdev_pidfile_write('%s') rc = %d\n", pidfile, rc);
        exit(2);
    }

    config = VDEV_CALLOC(struct vdev_config, 1);
    if (config == NULL)
        return -ENOMEM;
   
    rc = vdev_config_init(config);
    if (rc != 0) {
        vdev_error("vdev_config_init rc = %d\n", rc);
        goto Finish;
    }
    
    rc = vdev_config_load(config_file, config);
    if (rc != 0) {
        vdev_error("vdev_config_load('%s') rc = %d\n", config->config_path, rc);
        goto Finish;
    }
    
    config->config_path = vdev_strdup_or_null(config_file);
    if (!config->config_path) {  // OOM
        rc = -ENOMEM;
        goto Finish;
    }
    
    config->mountpoint = vdev_strdup_or_null(mountpoint);
    if (!mountpoint) {  // OOM
        rc = -ENOMEM;
        goto Finish;
    }
   
    cmd = VDEV_CALLOC(char, strlen(config->preseed_path) + 4 + strlen(mountpoint) + 2 + strlen(config_file) + 1);
    if (!cmd) {  // OOM
        rc = -ENOMEM;
        goto Finish;
    }
    
    /* 1 or 0 is irrelevant here */
    sprintf(cmd, "%s 1 %s %s", config->preseed_path, mountpoint, config_file);
    pfin = epopen(cmd, &pid);
    if (pfin) { 
           fclose(pfin);
           waitpid(pid, &wstatus, 0);
    }
    free(cmd);
    cmd = NULL;

    // ignore SIGPIPE from daemonlets 
    signal(SIGPIPE, SIG_IGN);

    daemonlet = VDEV_CALLOC(struct vdev_daemonlet, 1);
    if (daemonlet == NULL) {
        rc = -ENOMEM;
        goto Free;
    }
           
    // do we need to start it?
    if (daemonlet->pid <= 0) {
        rc = vdev_daemonlet_start(config,
                      daemonlet,
                      command,
                      name,
                      async,
                      &error_fd);
        if (rc < 0) {
            vdev_error("vdev_action_daemonlet_start('%s') rc = %d\n", name, rc);
            rc = -EPERM;
            goto Free;
        }
    }

    if (background)
        send_to_background();

    // Try to create a lock file 
    if (fd == -1) {
        fd = open(pidfile, O_RDWR|O_CREAT, 0640); 
        if (fd < 0) {
            switch(errno) {
            case ENOMEM:
                return -ENOMEM;
            default:
                return rc;
            }      
        }
    }

    memset(&fl, 0, sizeof(fl));
   
    fl.l_type = F_WRLCK; // Set a write lock
    fl.l_whence = SEEK_SET; // Start at beginning of file
    fl.l_start = 0; // Offset from beginning of file
    fl.l_len = 0; // Lock entire file
   
    // F_SETLK(W) ignores it; F_OFD_SETLK(W) requires it to be zero
    fl.l_pid = 0; //getpid(); // Set process ID
   
    // F_SETLKW specifies blocking mode
    fcntl(fd, F_SETLKW, &fl); // Set the lock

    sglib_cstr_vector_init(&uevents);
    rc = find_devices(sysfs_mountpoint, &uevents);

    bool is_tower_or_desktop = dmi_decode_type_is_tower_or_desktop();
    char *p[] = { "usb", "graphics", "drm", "pci", "i2c", "serio", "input", "hid", "hidraw", (char*)0 };
    for (unsigned i = 0; i < sizeof(p)/sizeof(char*) - 1; i++) {
        for (unsigned j = 0; j < sglib_cstr_vector_size(&uevents); j++) {
            char *subsystem = NULL;
            char *full_devpath = NULL;
            if (is_tower_or_desktop && strcmp(p[i], "input") && strcmp(p[i], "usb") && strcmp(p[i], "graphics") && strcmp(p[i], "pci"))
                continue;
            if (sglib_cstr_vector_at(&uevents, j) == NULL)
                continue;
            // extract the devpath from the uevent path 
            full_devpath = vdev_dirname(sglib_cstr_vector_at(&uevents, j), NULL);
            if (full_devpath == NULL)
                continue;
            rc = vdev_linux_sysfs_read_subsystem(sysfs_mountpoint, full_devpath, &subsystem);
            if (rc == 0) {
                if (subsystem && !strcmp(subsystem, p[i]) && !sysfs_get_parent_with_subsystem(sglib_cstr_vector_at(&uevents, j), "acpi")) {
                    feed_modprobe_daemonlet(sglib_cstr_vector_at(&uevents, j),
                                config,
                                daemonlet,
                                name,
                                command,
                                &error_fd);
                    sglib_cstr_vector_set(&uevents, NULL, j);
                }
                free(subsystem);
            }
            free(full_devpath);
        }
    }

    vdev_cstr_vector_free_all(&uevents);       
    sglib_cstr_vector_free(&uevents);

    if (fd > 0) {
        fl.l_type = F_UNLCK; // Release the lock
        fcntl(fd, F_SETLK, &fl); // Unlock the file
        close(fd);
    }
    unlink(pidfile);
   
    /* Netlink phase: wait for new uevent files */
   
    /* Initialize signals FD */    
    if ((ctx.pfd[FD_POLL_SIGNAL].fd = initialize_signals ()) < 0) {
        rc = -errno;
        vdev_error("Couldn't initialize signals: '%s'\n", strerror(errno));
        vdev_linux_context_free(&ctx);
        goto Free;
    } 
  
    /* Setup polling */
    ctx.pfd[FD_POLL_SIGNAL].events = POLLIN;
   
    /* Initialize signals FD */    
    if ((ctx.pfd[FD_POLL_FANOTIFY].fd = initialize_fanotify()) < 0) {
        rc = -errno;
        vdev_error("Couldn't initialize fanotify: '%s'\n", strerror(errno));
        vdev_linux_context_free(&ctx);
        goto Free;
    }

    /* Setup polling */
    ctx.pfd[FD_POLL_FANOTIFY].events = POLLIN;

    ctx.nl_addr.nl_family = AF_NETLINK;
    ctx.nl_addr.nl_pid = getpid();
    ctx.nl_addr.nl_groups = NETLINK_KOBJECT_UEVENT;
      
    ctx.pfd[FD_POLL_NETLINK].fd = socket(PF_NETLINK, SOCK_DGRAM, NETLINK_KOBJECT_UEVENT);
    if (ctx.pfd[FD_POLL_NETLINK].fd < 0) {
        rc = -errno;
        fprintf(stderr, "socket(PF_NETLINK) rc = %d\n", rc);
        vdev_linux_context_free (&ctx);
        return rc;
    }
      
    ctx.pfd[FD_POLL_NETLINK].events = POLLIN;
   
    // big receive buffer, if running as root 
    if (geteuid() == 0) {
        rc = setsockopt(ctx.pfd[FD_POLL_NETLINK].fd, SOL_SOCKET, SO_RCVBUFFORCE, &slen, sizeof(slen));
        if (rc < 0) {
            rc = -errno;
            fprintf(stderr, "setsockopt(SO_RCVBUFFORCE) rc = %d\n", rc);
            vdev_linux_context_free (&ctx);
            return rc;
        }
    }
  
    // check credentials of message--only root should be able talk to us
    rc = setsockopt(ctx.pfd[FD_POLL_NETLINK].fd, SOL_SOCKET, SO_PASSCRED, &so_passcred_enable, sizeof(so_passcred_enable));
    if (rc < 0) {
        rc = -errno;
        fprintf(stderr, "setsockopt(SO_PASSCRED) rc = %d\n", rc);
        vdev_linux_context_free(&ctx);
        return rc;
    }
    
    // bind to the address
    rc = bind(ctx.pfd[FD_POLL_NETLINK].fd, (struct sockaddr*)&ctx.nl_addr, sizeof(struct sockaddr_nl));
    if (rc != 0) {
        rc = -errno;
        fprintf(stderr, "bind(%d) rc = %d\n", ctx.pfd[FD_POLL_NETLINK].fd, rc);
        vdev_linux_context_free (&ctx);
        return rc;
    }

    while (1) {
        ssize_t len = 0;
        char uevent_buf[VDEV_LINUX_NETLINK_RECV_BUF_MAX];
        char cbuf[CMSG_SPACE(sizeof(struct ucred))];
        struct cmsghdr *chdr = NULL;
        struct ucred *cred = NULL;
        struct msghdr hdr;
        struct iovec iov;
        struct sockaddr_nl cnls;
   
        memset(&hdr, 0, sizeof(struct msghdr));

        // next event (wait forever)
        // NOTE: this is a cancellation point!
        rc = poll(ctx.pfd, FD_POLL_MAX, -1);
        if (rc < 0) {
            rc = -errno;
            if (rc == -EINTR) {
                // try again 
                continue;
            }
            vdev_error("FATAL: poll(%d) rc = %d\n", ctx.pfd[FD_POLL_NETLINK].fd, rc);
            break;
        }

        /* Signal received? */
        if (ctx.pfd[FD_POLL_SIGNAL].revents & POLLIN) {
            if (linux_signal == SIGTERM || linux_signal == SIGALRM || linux_signal == SIGINT ||  linux_signal == SIGQUIT)
                break;
            else
                continue;
        /* fanotify event received? */
        } else if (ctx.pfd[FD_POLL_FANOTIFY].revents & POLLIN) {
            char buffer[FANOTIFY_BUFFER_SIZE];
            ssize_t length;
            // Read from the FD. It will read all events available up to the given buffer size
            if ((length = read(ctx.pfd[FD_POLL_FANOTIFY].fd, buffer, FANOTIFY_BUFFER_SIZE)) > 0) {
                struct fanotify_event_metadata *metadata;
                metadata = (struct fanotify_event_metadata*)&buffer;
                while (FAN_EVENT_OK (metadata, length)) {
                    if (metadata->mask & FAN_Q_OVERFLOW) {
                        /* Queue overflow */
                        continue;
                    } else if (metadata->mask & FAN_OPEN) {
                        ssize_t linklen;
                        char fdpath[PATH_MAX + 1];
                        char path[PATH_MAX + 1];
                        char *tmp = NULL;           
                        sprintf(fdpath, "/proc/self/fd/%d", metadata->fd);
                        linklen = readlink(fdpath, path, sizeof(path) - 1);
                        if (linklen == -1) {
                            rc = -errno;
                            vdev_error("readlink(): '%s'\n", strerror(errno));
                            if (metadata->fd > 0)
                                close (metadata->fd);
                            break;
                        }
                        path[linklen] = '\0';
                        if (metadata->fd > 0)
                            close (metadata->fd);
                        if (!strcmp(path, "/bin/login") || !strcmp(path, "/usr/bin/Xorg")) {
                            vdev_linux_context_free(&ctx);
                            goto Free;
                        }
                    }
                    metadata = FAN_EVENT_NEXT(metadata, length);
                }
            }
            continue;           
        /* Netlink event received? */
        } else if (ctx.pfd[FD_POLL_NETLINK].revents & POLLIN) {
   
            // get the event 
            iov.iov_base = uevent_buf;
            iov.iov_len = VDEV_LINUX_NETLINK_RECV_BUF_MAX;
        
            hdr.msg_iov = &iov;
            hdr.msg_iovlen = 1;
   
            // get control-plane messages
            hdr.msg_control = cbuf;
            hdr.msg_controllen = sizeof(cbuf);
   
            hdr.msg_name = &cnls;
            hdr.msg_namelen = sizeof(cnls);

            // get the event 
            len = recvmsg(ctx.pfd[FD_POLL_NETLINK].fd, &hdr, 0);
            if (len < 0) {
                rc = -errno;
                fprintf(stderr, "FATAL: recvmsg(%d) rc = %d\n", ctx.pfd[FD_POLL_NETLINK].fd, rc);
                continue;
            }
   
            // big enough?
            if (len < 32 || len >= VDEV_LINUX_NETLINK_RECV_BUF_MAX) {
                fprintf(stderr, "Netlink message is %zd bytes; ignoring...\n", len);
                continue;
            }
   
            // control message, for credentials
            chdr = CMSG_FIRSTHDR(&hdr);
            if (chdr == NULL || chdr->cmsg_type != SCM_CREDENTIALS) {
                fprintf(stderr, "%s", "Netlink message has no credentials\n");
                continue;
            }
   
            // get the credentials
            cred = (struct ucred *)CMSG_DATA(chdr);
   
            // if not root, ignore 
            if (cred->uid != 0) {
                fprintf(stderr, "Ignoring message from non-root ID %d\n", cred->uid);
                continue;
            }
   
            // if udev, ignore... they are user space /dev events, and we are 
            // interested in driver core uevents that occur in the kernel space only 
            if (memcmp(uevent_buf, "libudev", 8) == 0) {
                // message from udev; ignore 
                fprintf(stderr, "%s", "Ignoring libudev message\n");
                continue;
            }
   
            // kernel messages don't come from userspace 
            if (cnls.nl_pid > 0) {
                // from userspace???
                fprintf(stderr, "Ignoring message from PID %d\n", (int)cnls.nl_pid);
                continue;
            }
   
            if (strncmp(uevent_buf, "add@/", 5)) {
                // invalid header 
                //fprintf(stderr, "%s", "invalid message header: missing '@' directive");
                continue;
            } 
   
            for (unsigned int i = 0; i < len;) {
                if (!strncmp(uevent_buf + i, "DEVPATH=", 8)) {
                    sbuf_t s;
                    char *tmp = rindex(uevent_buf + i, '=') + 1;
                    sbuf_init(&s);
                    sbuf_concat(&s, 2, tmp, "/uevent");
                    feed_modprobe_daemonlet(tmp,
                                config,
                                daemonlet,
                                name,
                                command,
                                &error_fd);
                    sbuf_free(&s);
                }
                i += strlen(uevent_buf + i) + 1;
            }
        }
    } // while
    
    vdev_linux_context_free(&ctx);

Free:    
    vdev_daemonlet_stop(daemonlet, name);
    
    if (daemonlet != NULL) {
        free(daemonlet);
        daemonlet = NULL;
    }

Finish:    
    if (config != NULL) {
        vdev_config_free(config);
        free(config);
        config = NULL;
    }

    return rc;
}
