/*
   vdev: a virtual device manager for *nix
   Copyright (C) 2023  Aitor C.Z.

   This program is dual-licensed: you can redistribute it and/or modify
   it under the terms of the GNU General Public License version 3 or later as 
   published by the Free Software Foundation. For the terms of this 
   license, see LICENSE.GPLv3+ or <http://www.gnu.org/licenses/>.

   You are free to use this program under the terms of the GNU General
   Public License, but WITHOUT ANY WARRANTY; without even the implied 
   warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
   See the GNU General Public License for more details.

   Alternatively, you are free to use this program under the terms of the 
   Internet Software Consortium License, but WITHOUT ANY WARRANTY; without
   even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
   For the terms of this license, see LICENSE.ISC or 
   <http://www.isc.org/downloads/software-support-policy/isc-license/>.
*/

#include "misc.h"
#include "sbuf.h"

#include <assert.h>
#include <pstat/libpstat.h>
#include <utmp.h>   ////

// Usage: strncpy_t(str1, sizeof(str1), str2, strlen(str2));
// Copy string "in" with at most "insz" chars to buffer "out", which
// is "outsz" bytes long. The output is always 0-terminated. Unlike
// strncpy(), strncpy_t() does not zero fill remaining space in the
// output buffer:
char *strncpy_t(char *out, size_t outsz, const char *in, size_t insz)
{
    assert(outsz > 0);
    while (--outsz > 0 && insz > 0 && *in) {
        *out++ = *in++;
        insz--;
    }
    *out = 0;
    return out;
}

void kill_previous_instances(const char *binary)
{
	DIR *dirp = NULL;
	struct dirent *dir = NULL;
    struct pstat *ps = NULL;
    int rc = 0;
       
    pid_t pid = getpid();
    
	ps = pstat_new();
	if (!ps) {
		vdev_error("pstat(%d) %s\n", pid, strerror(errno));
		pstat_free(ps);
		exit(8);
	}
		
	rc = pstat(pid, ps, 0);
	if (rc != 0) {
		vdev_error("pstat(%d) %s\n", pid, strerror(errno));
		pstat_free(ps);
		exit(9);
	}
	
	dirp=opendir("/proc");
	if (!dirp) {
		vdev_error("opendir() %s\n", strerror(errno));
		exit(10);
	}
	
	while ((dir=readdir(dirp)) != NULL) {
        int rc = 0;
        struct stat sb;
        char *proc_cmd = NULL;
        bool is_uint_ok = true;
        char pth[NAME_MAX + 1] = {0};
        pid_t proc_pid = 0;
	    
        for (unsigned i=0; i<strlen(dir->d_name); i++) {
            if (!isdigit(dir->d_name[i]))
                is_uint_ok = false;
        }
    
        if (!is_uint_ok)
            continue;
    
        strncpy_t(pth, sizeof(pth), "/proc/", strlen("/proc/"));
        strcat(pth, dir->d_name);
        if (stat(pth, &sb) == -1)
            continue;
    
        if (!lstat(dir->d_name, &sb))
            continue;
 
        /* Only consider directories */
        if (!S_ISDIR(sb.st_mode))
            continue;
		
        rc = sscanf(dir->d_name, "%d", &proc_pid);
        if (rc != 1) {
            vdev_error("sscanf(%d) rc = %d\n", proc_pid, rc);
            pstat_free(ps);
            continue;
        }

        /* Only foreing processes */
        if (pid == proc_pid)
            continue;

        rc = pstat_get_binary_path_from_string(&proc_cmd, NAME_MAX + 1 , dir->d_name);
        if (rc)
            continue;

        if (proc_cmd && !strcmp(rindex(proc_cmd, '/') + 1, binary)) {
    
			struct pstat *proc_ps = NULL;
    
			proc_ps = pstat_new();
			if (!proc_ps)
                continue;
		
			rc = pstat(proc_pid, proc_ps, 0);
			if (rc != 0) {
				vdev_error("pstat(%d) rc = %d\n", proc_pid, rc);
				pstat_free(proc_ps);
				continue;
			}
    
			/* Only previous processes */
			if (pstat_get_starttime(ps) <= pstat_get_starttime(proc_ps)) {
				pstat_free(proc_ps);
				continue;
			}
			
			pstat_free(proc_ps);
			
			kill(proc_pid, SIGTERM);
        }

        free(proc_cmd);
	}
	
    pstat_free(ps);
    closedir(dirp);
}

bool is_running(const char *binary)
{
	DIR *dirp = NULL;
	struct dirent *dir = NULL;
    struct pstat *ps = NULL;
    int rc = 0;
    bool is_running_ok = false;
       
    pid_t pid = getpid();
    
	ps = pstat_new();
	if (!ps) {
		vdev_error("pstat(%d) %s\n", pid, strerror(errno));
		pstat_free(ps);
		exit(8);
	}
		
	rc = pstat(pid, ps, 0);
	if (rc != 0) {
		vdev_error("pstat(%d) %s\n", pid, strerror(errno));
		pstat_free(ps);
		exit(9);
	}
	
	dirp = opendir("/proc");
	if (!dirp) {
		vdev_error("opendir() %s\n", strerror(errno));
		exit(10);
	}
	
	while ((dir=readdir(dirp)) != NULL) {
        int rc = 0;
        struct stat sb;
        char *proc_cmd = NULL;
        bool is_uint_ok = true;
        sbuf_t pth;
        pid_t proc_pid = 0;
	    
        for (unsigned i=0; i<strlen(dir->d_name); i++) {
            if (!isdigit(dir->d_name[i]))
                is_uint_ok = false;
        }
    
        if (!is_uint_ok)
            continue;
    
        sbuf_init(&pth);
        sbuf_concat(&pth, 2, "/proc/", dir->d_name);
        if (stat(pth.buf, &sb) == -1) {
            sbuf_free(&pth);
            continue;
        }
        sbuf_free(&pth);
    
        if (!lstat(dir->d_name, &sb))
            continue;

        /* Only consider directories */
        if (!S_ISDIR(sb.st_mode))
            continue;
		
		rc = sscanf(dir->d_name, "%d", &proc_pid);
        if (rc != 1) {
            vdev_error("sscanf(%d) rc = %d\n", proc_pid, rc);
            pstat_free(ps);
            continue;
        }

        /* Only foreing processes */
        if (pid == proc_pid)
            continue;

		rc = pstat_get_binary_path_from_string(&proc_cmd, NAME_MAX + 1 , dir->d_name);
		if (rc)
            continue;

        if (proc_cmd && !strcmp(rindex(proc_cmd, '/') + 1, binary)) {
            struct pstat *proc_ps = NULL;
			proc_ps = pstat_new();
			if (!proc_ps)
                continue;
		
			rc = pstat(proc_pid, proc_ps, 0);
			if (rc != 0) {
				vdev_error("pstat(%d) rc = %d\n", proc_pid, rc);
				pstat_free(proc_ps);
				continue;
			}
    
			/* Only previous processes */
			if (pstat_get_starttime(ps) <= pstat_get_starttime(proc_ps)) {
				pstat_free(proc_ps);
				continue;
			}
			
			is_running_ok = true;
			pstat_free(proc_ps);
        }
        free(proc_cmd);
    }
	
    pstat_free(ps);
    closedir(dirp);
    
    return is_running_ok;
}

// thread-safe and async-safe int to string for base 10
void itoa10_safe(int val, char *str)
{
    int i = 0;
    int len = 0;
    int j = 0;
    bool neg = false;
   
    /* sign check */
    if (val < 0) {
        val = -val;
        neg = true;
    }
   
    /* consume, lowest-order to highest-order */
    if (val == 0) {
        str[i] = '0';
        i++;
    } else {
        while (val > 0) {
            int r = val % 10;
            str[i] = '0' + r;
            i++;
            val /= 10;
        }
    }
   
    if (neg) {
        str[i] = '-';
        i++;
    }
   
    len = i;
    i--;
   
    /* reverse order to get the number */
    while (j < i) {
        char tmp = *(str + i);
        *(str + i) = *(str + j);
        *(str + j) = tmp;
        j++;
        i--;
    }
   
    str[len] = '\0';
}

/**
 * \brief A subprogram like popen() but with diferences explained below
 * The pipe is to read in the current process and connected to stderr in the
 * new one. The intent is to report error messages. Plus pid is returned.
 * Disclaimer: The caller must close the pipe after use
 */
FILE *epopen(const char *cmd,  pid_t *pid)
{
    int pipefd[2];
    pid_t p_id;
    FILE *pf;
    int rc;

    /* Create the pipe */
    if (pipe(pipefd)) {
        *pid = -1;
        return NULL;
    }

    p_id = fork();
 
    switch(p_id) {
    case 0: /* child */
        close(pipefd[0]);
        close(STDERR_FILENO);
        dup2(pipefd[1], STDERR_FILENO);
        close(pipefd[1]);
        execl("/bin/sh", "sh", "-c", cmd, (char *)NULL);
        fprintf(stderr, "error invoking /bin/sh: %s\n", strerror(errno));
        exit(EXIT_FAILURE); /* reminder: we are the child */
    case -1: /* fork() failed ! */
        rc = errno;
        close(pipefd[0]);
        close(pipefd[1]);
        errno = rc;
        *pid = -1;
        return NULL;
    default: /* the parent */
        close(pipefd[1]);
        pf = fdopen(pipefd[0], "r");
        *pid = p_id;
        return pf;
    }
}


void send_to_background()
{
    int ret;
    pid_t pid = 0;
    int fd;

    /* Fork off the parent process.
     * The first fork will change our pid
     * but the sid and pgid will be the
     * calling process */
    pid = fork();

    /* An error occurred */
    if (pid < 0)
        exit(EXIT_FAILURE);

    /* Fork off for the second time.
     * The magical double fork. We're the session
     * leader from the code above. Since only the
     * session leader can take control of a tty
     * we will fork and exit the session leader.
     * Once the fork is done below and we use
     * the child process we will ensure we're
     * not the session leader, thus, we cannot
     * take control of a tty. */
    if (pid > 0)
        exit(EXIT_SUCCESS);

    /* On success: The child process becomes session leader */
    if (setsid() < 0)
        exit(EXIT_FAILURE);

    /* Fork off for the second time */
    pid = fork();

    /* An error occurred */
    if (pid < 0)
        exit(EXIT_FAILURE);

    /* Success: Let the parent terminate */
    if (pid > 0)
        exit(EXIT_SUCCESS);
   
    /* Set new file permissions */
    umask(0);

    /* Change the working directory to /tmp
     * or another appropriated directory
     */
    ret = chdir("/tmp");
    if (ret != 0)
        exit(EXIT_FAILURE);
}


// get the parent device of a given device, using sysfs.
// not all devices have parents; those that do will have a uevent file.
// walk up the path until we reach /
// return 0 on success
// return -EINVAL if the dev path has no '/'
// return -ENOENT if this dev has no parent
// return -errno if stat'ing the parent path fails 
// return -ENOMEM if OOM
int sysfs_get_parent_device(char const *dev_path, 
                            char **ret_parent_device, size_t *ret_parent_device_len)
{
    char *parent_path = NULL;
    size_t parent_path_len = 0;
    struct stat sb;
    int rc = 0;
   
    char *delim = NULL;
   
    parent_path = (char*)calloc(strlen(dev_path) + 1 + strlen("/uevent"), 1);
    if (!parent_path)
        return -ENOMEM;
   
    strcpy(parent_path, dev_path);
   
    while (strlen(parent_path) > 0) {
        /* lop off the child */ 
        delim = rindex(parent_path, '/');
        if (!delim) {
            /* invalid */ 
            free(parent_path);
            return -EINVAL;
        }
      
        if (delim == parent_path) {
            /* reached (/) */
            free(parent_path);
            return -ENOENT;
        }
      
        while (*delim == '/' && delim != parent_path) {
            *delim = '\0';
            delim--;
        }
      
        parent_path_len = strlen(parent_path);
      
        /* verify that this is a device... */
        strcat(parent_path, "/uevent");
      
        rc = stat(parent_path, &sb);
        if (rc != 0) {
            /* not a device
             * remove /uevent
             */ 
            parent_path[parent_path_len] = '\0';
            continue;
        } else {
            /* device! */
            break;
        }
    }
   
    /* lop off /uevent */ 
    parent_path[parent_path_len] = '\0';
   
    *ret_parent_device = parent_path;
    *ret_parent_device_len = parent_path_len;
   
    return 0;
}

bool sysfs_get_parent_with_subsystem(char const *device_path, 
                                     char const *subsystem_name)
{
    bool res = false;
    char cur_dir[4097];
    char subsystem_path[4097];
    char subsystem_link[4097];
   
    memset(subsystem_path, 0, 4097);
    memset(subsystem_link, 0, 4097);
   
    char *tmp = NULL;
    int rc = 0;
    char *parent_device = NULL;
    size_t parent_device_len = 0;
   
    strcpy(cur_dir, device_path);
   
    while (1) {
        /* get parent device */ 
        rc = sysfs_get_parent_device(cur_dir, &parent_device, &parent_device_len);
        if (rc != 0)
            break;
      
        /* subsystem? */
        sprintf(subsystem_path, "%s/subsystem", parent_device);
      
        memset(subsystem_link, 0, 4096);
      
        rc = readlink(subsystem_path, subsystem_link, 4096);
        if (rc < 0) {
            rc = -errno;
            if (rc != -ENOENT)
                fprintf(stderr, "[WARN]: readlink('%s') errno = %d\n", 
                                subsystem_path, rc);
     
            free(parent_device);
            parent_device = NULL;
            return res;
        }
      
        // get subsystem name...
        tmp = rindex(subsystem_link, '/');
        if (tmp) {
            if (strcmp(tmp + 1, subsystem_name) != 0) {
                /* subsystem does not match
                 * crawl up to the parent
                 */ 
                strcpy(cur_dir, parent_device);
                free(parent_device);
                parent_device = NULL;
                continue;
            } else {
                res = true;
                break;
            }
        } else {
            break;
        }
    }
  
    return res;
}

int get_user_sessions()
{
    struct utmp *u;
    int count = 0;
    
    setutent();
    while ((u = getutent()) != NULL) {
        if (u->ut_line[0] == '~')
            continue;
        if (u->ut_user[0] && (strcmp(u->ut_user, "LOGIN")))
            count++;
    }
    endutent();

    return count;
}
